
import matplotlib.pyplot as plt
import numpy as np

import os
import sys
sys.path.append("..") # for TransientImage package
import TransientImage
import ApplyNoise

# Set this to an test file from the data base!
originalFilename = "FlyingDragon.ti"
noisyFilename = "FlyingDragonNoise.ti"

def PlotTile(name, data, vmin, vmax, ext):
	plt.figure(name)
	plt.imshow(data, vmin=vmin, vmax=vmax)
	plt.xticks([])
	plt.yticks([])
	plt.tight_layout()
	plt.gcf().set_size_inches(3, 3)
	plt.savefig(name+ext)

def StreakPlot(ti, name, tstart, tend, p0, p1, p2):
	ext = '.png'
	
	vmin=0
	vmax=np.percentile(ti.data[:,:,p2], 99)
	
	plt.figure(name+"-side")
	plt.imshow(ti.data[128, :, tstart:tend], aspect='auto', vmin=vmin, vmax=vmax)
	plt.xticks([])
	plt.yticks([])
	plt.plot((p0-tstart, p0-tstart), (0, 255), color="r")
	plt.plot((p1-tstart, p1-tstart), (0, 255), color="r")
	plt.plot((p2-tstart, p2-tstart), (0, 255), color="r")
	plt.gcf().set_size_inches(9, 3)
	plt.tight_layout()
	plt.savefig(name+'-streak'+ext)
	
	PlotTile(name+'-P0', ti.data[:,:,p0], vmin, vmax, ext)
	PlotTile(name+'-P1', ti.data[:,:,p1], vmin, vmax, ext)
	PlotTile(name+'-P2', ti.data[:,:,p2], vmin, vmax, ext)
	
	
def PlotPhaseImage(name, data, vmax):
	ext = '.pdf'
	plt.figure(name)
	plt.imshow(data, cmap="bwr", vmin=-vmax, vmax=vmax)
	plt.tight_layout()
	plt.xticks([])
	plt.yticks([])
	plt.colorbar()
	plt.gcf().set_size_inches(4, 3)
	plt.savefig(name+ext)
	
	
################################################
	
def PlotPhaseNoise():
	ti = TransientImage.TransientImage(noisyFilename)
	pA = ApplyNoise.ApplyPhaseFunction(ti, "40Mhz.phase0calib")
	pB = ApplyNoise.ApplyPhaseFunction(ti, "40Mhz.phase1calib")
	
	maxV = max(np.max(np.abs(pA)), np.max(np.abs(pB)))
	
	PlotPhaseImage('PhaseA', pA, maxV)
	PlotPhaseImage('PhaseB', pB, maxV)
	
	# plot phase function
	plt.figure('Phase functions')
	plt.plot(np.fromfile("40Mhz.phase0calib", dtype=np.float32), label='Phase A')
	plt.plot(np.fromfile("40Mhz.phase1calib", dtype=np.float32), label='Phase B')
	plt.xticks([0, 127, 255], ['0', '$\pi$', '$2\pi$'])
	plt.legend()
	plt.tight_layout()
	plt.gcf().set_size_inches(6, 3)
	plt.grid()
	plt.savefig('PhaseFunctions.pdf')
	
def PlotSpadNoise():
	tstart = 180
	tend = 300
	p0 = 200
	p1 = 220
	p2 = 270

	StreakPlot(TransientImage.TransientImage(originalFilename), 'Original',
			tstart, tend, p0, p1, p2)
	StreakPlot(TransientImage.TransientImage(noisyFilename), 'Noise',
			tstart, tend, p0, p1, p2)
	
if __name__ == "__main__":
	if not os.path.isfile(originalFilename):
		raise Exception("please provide a file from the data set")
	
	if not os.path.isfile(noisyFilename):
		ApplyNoise.ApplyNoiseToFile(originalFilename, 1000000, 0.7, 2.3, noisyFilename)
		

	plt.close('all')
	
	PlotPhaseNoise()
	PlotSpadNoise()
	
	plt.close('all')
	print('done')